/*
==== PinDown JavaScript tools 20250828 ====
= Roel Buining - LuxSoft - www.LuxSoft.eu =
*/

'use strict';

//shortcut function
function $I(id) { return document.getElementById(id); }
function $N(name) { return document.getElementsByName(name); }

/*===== global variables =====*/
//in <dbname>.js file
// var m: marker image field indexes
// var markers: marker definitions (each marker: array with images; the first image is the marker image)
//in <dbname>.ext.js file
// various config setting variables
// var cf: general config settings

//pindown.php
var map; //the map object
const mkrIcons = {}; //marker icons
//toolbox.js - volatile
var mOnMap = []; //markers currently shown
var clusters; //marker clusters object
var rCircle; //radius circle object
var fCircle; //fly-to circle object
var gWatch; //GPS watcher (0: off, 1: on)
var gWatchID; //GPS watcher ID
var gCircle; //GPS circle object
var gMarker; //GPS marker object
var gAutoCr = false; //GPS automatic centering when location updated
var gCrOnce = false; //GPS centered once after ON 
var markerI; //marker index when marker selected
var iBoxImI; //info box marker image index
var folding; //sub-images in displayed marker set
var pinLayer; //layer ID of combi-view pin
var usrPins = []; //map click user pins (layer, lat, lng)
var usrPoly; //map click polylines layer
var rteNode; //node icon for user routes
const narWin = window.matchMedia("(max-width:700px)").matches;
const featCols = {}; //loaded feature collections (geoJSON fName: content)
const fLayers = {}; //active feature collections (fName: layer)
const rLayers = {}; //active external routes (fName: layer)
const ulRoutes = {}; //loaded external routes (geoJSON fName: content)
//toolbox.js - saved
let S = {}; //saved variables object
const defS = { //default values of S
	controls: structuredClone(cf._controls), //control selection defaults
	mlView: narWin ? 'map' : cf._defView, //view "map"|"list"|"both"
	mLat: cf._defLat, //map center Latitude
	mLon: cf._defLng, //map center Longitude
	rRad: cf._defRadius, //radius (in km)
	mZoom: cf._defZoom, //map zoom factor
	cBox: 0, //control box (0: hidden, 1: visible)
	cBoxX: "12", //control box X-pos
	cBoxY: "110", //control box Y-pos
	iBox: -1, //info box (-1: hidden, #: marker index or string: image file name)
	iBoxX: "12", //info box X-pos
	iBoxY: narWin ? "160" : "450", //info box Y-pos
	fGroup: cf._defGroup, //filter marker group(s)
	fType: _key1Start, //filter marker type
	fTags: [], //filter on tags
	fTsOA: 0, //filter tags AND|OR (0: OR, 1: AND)
	key2Min: "", //filter key 2 min
	key2Max: "", //filter key 2 max
	dateFr: "", //filter from date
	dateTo: "", //filter to date
	sMenuID: "", //currently active side menu ID (empty: no sm active)
	actGeo: [] //active geoJSON file names
};

//onload run these functions
window.onload = (event) => {
	const flytoURL = window.location.href.match(/flyto=(\d{1,3}.\d{4,6}),(\d{1,3}.\d{4,6})(.*)/);
	if (flytoURL) { //URL flyto
		const zoom = flytoURL[3].match(/,Z(\d{1,2})/i);
		const type = flytoURL[3].match(/,T(\d{1,2})/i);
		flytoURL[3] = zoom ? zoom[1] : cf._ftZoom;
		initVars(0);
		S.fType = type ? type[1] : _key1Start;
	} else {
		let init = (typeof snapS !== 'undefined') ? 1 : ((cf._lastSel && localStorage.getItem("remS") == '1') ? 2 : 0);
		$I("rememSet").checked = init == 2 ? true : false;
		initVars(init);
	}
	siteInfo();
	initCtls();
	initMkrs();
	initMap();
	initCbox();
	initIbox();
	if (S.sMenuID) {
		showSm(S.sMenuID);
	}
	showGeoJson();
	pageView(S.mlView);
	if (flytoURL) { //URL flyto
		flyToLL('u',flytoURL[1],flytoURL[2],flytoURL[3]);
	}
};

//on tab close save Map Control vars to localStorage
document.onvisibilitychange = () => {
	if (document.visibilityState === "hidden") {
		localStorage.setItem("S",JSON.stringify(S));
	}
};

//make snapshot of map control settings
async function mcSnapShot() {
	const result = await askServer('set',S);
	const response = result.ok != 'OK' ? result.ok : `Snapshot saved\n${result.info}`;
	alert(response);
}

/*===== detect click anywhere outside =====*/
window.onclick = function(event) { //when click outside . . .
	if (event.target != locList) { $I('locList').innerHTML = ''; } //close location list
	if (event.target == modal) { hide('modal'); } //close modal
}

//total application reset
function totReset() {
	const actGeo = [...S.actGeo];
	actGeo.forEach(value => geoJSON(value)); //remove all routes
	initVars(0); //reset all variables to their default value
	//remove circles and GPS marker
	if (fCircle) { map.removeLayer(fCircle); }
	if (rCircle) { map.removeLayer(rCircle); }
	if (gCircle) { map.removeLayer(gCircle); }
	if (gMarker) { map.removeLayer(gMarker); }
	closeUdR('r');
	$I('fndInp').value = '';
	pageView(S.mlView);
	initCtls();
	//populate map and list
	initCbox();
	initIbox();
	toggleUlRs();
	setView(S.mLat,S.mLon,S.mZoom);
	addCircle(S.mLat,S.mLon,S.rRad);
	addMarkers(S.mLat,S.mLon,S.rRad);
}

//get vars from snapshot, localStorage or defaults
function initVars(init) {
	if (init == 2 && localStorage.getItem("S")) { //from local storage
		S = JSON.parse(localStorage.getItem("S"));
	} else if (init == 1) { //from snapshot file
		cf._lastSel = 0;
		if (narWin) {
			snapS.mlView = 'map';
		}
		S = snapS;
	} else { //defaults
		S = JSON.parse(JSON.stringify(defS)); //deep clone defS
	}
	if (init > 0) { //ensure S is complete
		for (const prop in defS) {
			if (prop in S === false) { S.prop = defS.prop; }
		}
	}
}

//initialize controls
function initCtls() {
	const setMenu = $I('setMenu');
	for (const [ctrlID, value] of Object.entries(S.controls)) {
		$I(ctrlID).style.display = value ? 'inline-block' : '';
		if (value && setMenu.querySelector(`.${ctrlID}`)) { //update side menu check boxes
			setMenu.querySelector(`.${ctrlID}`).checked = true;
		}
	}
}

//toggle full screen mode
function fullScreen() {
	if (!document.fullscreenElement) {
			document.documentElement.requestFullscreen();
	} else {
		document.exitFullscreen();
	}
}

//remove element from DOM
function hide(elmID) {
	$I(elmID).style.display = 'none';
}

//toggle tag list display
function toggleTags(elmID) {
	const elm = $I(elmID);
	if (elm.style.display == 'block') {
		elm.style.display =  '';
	} else {
		$N('fTsOA')[S.fTsOA].checked = true; //restore OR | AND
		$N('tags[]').forEach(node => { node.checked = S.fTags.includes(node.value); }); //restore checked tags
		elm.style.display =  'block';
	}
}

//submit selected tags
function submitTags(tags,OorA) {
	S.fTsOA = $N(OorA)[0].checked ? 0 : 1; //0: OR, 1: AND
	S.fTags.length = 0; //init array
	$N(tags).forEach(node => { if (node.checked) S.fTags.push(node.value); }); //save checked tags
	$I('active').style.display = S.fTags.length > 0 ? 'inline' : '';
	addMarkers(S.mLat,S.mLon,S.rRad);
}

//await fetch from server function
async function askServer(query,data) {
	data.q = query;
	const response = await fetch(`includes/toolboxaf.php`, {
		method: 'POST',
		mode: "same-origin",
		credentials: "same-origin",
		headers: { 'Content-Type':'application/json' },
		body: encodeURIComponent(JSON.stringify(data)), //encode json data
	});
	if (!response.ok) { alert(`Fetch error! status: ${response.status}`); return; }
	return await response.json();
}

//show initial site info box
function siteInfo() {
	let xBox = $I("xBox");
	if ($I("xBoxText").innerHTML.length < 50 || localStorage.getItem("siInfo") !== null) { //empty or shown
		xBox.style.visibility = "hidden";
	} else {
		xBox.style.opacity = 1;
	}
}

//toggle site info box
async function xToggle(file) {
	let info = '';
	if ($I("xBox").style.opacity == 0) { //show info
		const result = await askServer('sin',{file:file});
		if (result.ok != 'OK') { alert(`Fetch error: ${result.ok}`); }
		info = result.info;
	}
	toggleXbox(info);
}

//toggle controls in settings menu
function toggleCtrl(toggle,ctrlID) {
	$I(ctrlID).style.display = toggle.checked ? 'inline' : '';
	S.controls[ctrlID] = toggle.checked ? 1 : 0;
}

//toggle a side menu route group
function toggleDisplay(div) {
	const allGroups = $I('lyrMenu').querySelectorAll(".lyrGroup");
	const actGroup = $I(div);
	for (let i = 0; i < allGroups.length; i++) {
		if (allGroups[i] != actGroup) {
			allGroups[i].style.display = '';
		}
	}
	actGroup.style.display = actGroup.style.display == '' ? 'block' : '';
}

//clear input field
function clearFld(fID) { 
	$I(fID).value = '';
}

//(un)fold side menus
function showSm(menuID) {
	const mIDs = ['app','lyr','udr','set'];
	mIDs.forEach((mID) => {
		let mElm = $I(`${mID}Menu`);
		if (mID == menuID) {
			mElm.style.width = (mElm.clientWidth > 50) ? "0px" : `${cf._sMenuWidth}px`;
			S.sMenuID = (mElm.clientWidth > 50) ? "" : mID;
		} else {
			mElm.style.width = "0px";
		}
	});
}

//show/hide map control box
function showCbox() {
	$I("cBox").style.display = S.cBox ? '' : 'block';
	S.cBox = S.cBox == 0 ? 1 : 0;
	$I('cBox').style.zIndex = 1000;
}

//position and fill cBox after page reload
function initCbox() {
	let box = $I("cBox");
	box.style.display = S.cBox ? 'block' : '';
	box.style.left = S.cBoxX + 'px';
	box.style.top = S.cBoxY + 'px';
	$I('place').value = '';
	$I('cLat').value = S.mLat;
	$I('cLng').value = S.mLon;
	$I("rRad").value = S.rRad;
	$I('mcEvents').style.display = S.rRad > 0 ? 'inline' : 'none';
	$I('active').style.display = S.fTags.length > 0 ? 'inline' : '';
	if ($I('fGroup')) {
		const grpObj = $I('fGroup');
		grpObj.value = S.fGroup;
		$I('gLayer').style.display = (grpObj.options[grpObj.selectedIndex].dataset.gjfile) ? 'inline' : '';
	}
	if ($I('fType')) { 
		$I('fType').value = S.fType;
		showInfoBut(S.fType);
	}
	$I('rememSet').style.display = cf._lastSel ? 'inline' : 'none';

	box.addEventListener('touchmove', function(e) {
		const touchLocation = e.targetTouches[0]; //touch location
		box.style.left = (touchLocation.pageX - 125) + 'px';
		box.style.top = (touchLocation.pageY - 75) + 'px';
	})
}

//position iBox after page reload
function initIbox() {
	let box = $I("iBox");
	box.style.left = S.iBoxX + 'px';
	box.style.top = S.iBoxY + 'px';
	box.addEventListener('touchmove', function(e) {
		const touchLocation = e.targetTouches[0]; //touch location
		box.style.left = (touchLocation.pageX - 150) + 'px';
		box.style.top = (touchLocation.pageY - 75) + 'px';
	})
	if (S.iBox !== -1) {
		showIbox(0,S.iBox);
	} else {
		hideIbox();
	}
}

function showInfoBut(fType) {
	let mTypePars = _key1Val[fType].split(';');
	if (mTypePars.length > 1) {
		let infoBut = $I('infoBut');
		infoBut.title = mTypePars[1];
		const target =	mTypePars[3] ?? '_blank';
		infoBut.onclick = function() {
			window.open(mTypePars[2],target);
		};
		infoBut.style.display = 'inline';
	}
}

//hide marker info box
function hideIbox() { 
	hide('iBox');
	S.iBox = -1;
}

//toggle map/list view (mem | map | list | both)
function pageView(view,cur = '') { //go to view if the current view is cur
	if (cur && S.mlView != cur) { return; }
	if (view === 'mem') {
		view = S.mlView;
	}
	if (narWin && view == 'both') {
		view = 'map';
	}
	S.mlView = $I('mlcBut').value = view;
	let map = $I('map');
	if (view === "list") { //go to list view
		hideIbox(); //close info box
		$I('gpsBut').style.visibility = "hidden"; //hide GPS button
		$I('ggmBut').style.visibility = "hidden"; //hide G-map button
		map.style.zIndex = '900';
		$I('cBox').style.zIndex = 0;
		$I('iBox').style.zIndex = 0;
	} else { //go to map or combi view
		map.style.left = view === "map" ? '' : '363px'; //mape or combi view
		map.style.zIndex = '902';
		$I('gpsBut').style.visibility = ""; //show GPS button
		$I('ggmBut').style.visibility = ""; //show G-map button
		$I('cBox').style.zIndex = 1000;
		$I('iBox').style.zIndex = 1000;
	}
}

//toggle map
function toggleMap(lat,lng) {
	if ($I('map').style.zIndex == '902') {
		$I('map').style.zIndex = '900';
		S.mlView = $I('mlcBut').value = 'list';
		$I('cBox').style.zIndex = 0;
		$I('iBox').style.zIndex = 0;
	} else {
		$I('map').style.zIndex = '902';
		$I('map').style.left = '363px';
		S.mlView = $I('mlcBut').value = 'both';
		if (pinLayer) { map.removeLayer(pinLayer);}
		if (lat && lng) {
			setView(lat,lng);
			pinLayer = L.marker([lat,lng],{icon:mkrIcons['pin'], pane:'pinPane'}).addTo(map);
			pinLayer.on('click', function() { removePin(pinLayer); });
		}
		$I('cBox').style.zIndex = 1000;
		$I('iBox').style.zIndex = 1000;
	}
}

/*===== geoJSON functions =====*/

//show remembered geoJSON features
function showGeoJson() {
	const actGeo = [...S.actGeo];
	S.actGeo = [];
	actGeo.forEach(value => geoJSON(value,1)); //show
}

//toggle display of geoJSON feature
async function geoJSON(featCol,forceOn = 0) {
	if (!fLayers[featCol]) { //show object
		if (!featCols[featCol]) { //retrieve geojson file from server
			const result = await askServer('gjs',{fName:featCol});
			if (result.ok != 'OK') {
				alert(`geoJSON load error: ${result.ok}`);
				return; //no geoJSON file
			}
			const geoFile = JSON.parse(result.geoFile);
			featCols[featCol] = geoFile.contents;
		}
		const pane = featCols[featCol].pane == 'area' ? 'areaPane' : 'overlayPane';
		const colProps = featCols[featCol].properties ?? {};
		fLayers[featCol] = L.geoJSON(featCols[featCol], {
			style: function (feature) {
				return getGeoStyle(feature,colProps);
			},
			onEachFeature: function (feature, layer) {
				if (feature.properties.description) {
					layer.bindPopup(feature.properties.description);
				}
				if (feature.properties.distance) {
					layer.bindTooltip(feature.properties.distance, {direction:'right', permanent:true, offset: [-4,0], className: "xKm"});
				} else if (feature.properties.name) {
					layer.bindTooltip(feature.properties.name);
				}
			},
			pointToLayer: function (feature, latlng) {
        return new L.circleMarker(latlng);
			}, pane:pane
		}).addTo(map);
		S.actGeo.push(featCol);
	} else if (!forceOn) { //remove object
		map.removeLayer(fLayers[featCol]);
		delete fLayers[featCol];
		S.actGeo.splice(S.actGeo.indexOf(featCol),1);
	}
	updChBoxes();
}

//get the style for the a geojson feature
function getGeoStyle(feature,colProps) {
	let style = {};
	const props = feature.properties ?? {};
	style.radius = props.radius ?? colProps.radius ?? cf._gjRadius;
	style.color = props.stroke ?? colProps.stroke ?? cf._gjColor;
	style.weight = props['stroke-width'] ?? colProps['stroke-width'] ?? cf._gjWidth;
	style.opacity = props['stroke-opacity'] ?? colProps.opacity ?? cf._gjOpacity;
	style.fillColor = props.fill ?? colProps.fill ?? cf._gjFill;
	style.fillOpacity = props['fill-opacity'] ?? colProps['fill-opacity'] ?? cf._gjFillOpacity;
	return style;
}

//from side menu label
function geoJSONsm(gjFile,lat,lng,zoom) {
	pageView('map','list');
	setView(lat,lng,zoom);
	if (!S.actGeo.includes(gjFile)) {
		geoJSON(gjFile); //show route
	}
}

//show route - from list view
function geoJSONlv(markerI,gjFile,zoom = cf._defZoom) {
	const lat = markers[markerI][0][m.lat];
	const lng = markers[markerI][0][m.lng]
	pageView('both','list');
	setView(lat,lng,zoom);
	geoJSON(gjFile,1); //show route
	showIbox(0,markerI); //show marker info
}

//toggle group selection route
function lyrToggle() {
	const grpObj = $I('fGroup');
	const gjfile = grpObj.options[grpObj.selectedIndex].dataset.gjfile;
	geoJSON(gjfile); //toggle layers
}

//update route check boxes
function updChBoxes() {
	//update marker group checkbox
	const grpObj = $I('fGroup');
	const gjfile = grpObj.options[grpObj.selectedIndex].dataset.gjfile;
	$I('grCheckBox').checked = S.actGeo.includes(gjfile) ? true : false;
	//update side menu checkboxes
	const cBoxes = $I('lyrMenu').querySelectorAll(".smCBox");
	for (let i = 0; i < cBoxes.length; i++) {
		const div = cBoxes[i].parentElement; //parent div
		const hit = S.actGeo.includes(div.dataset.gjfile);
		cBoxes[i].checked = hit;
		//enable layer info
		const lyrInfo = div.querySelector(".lyrInfo");
		if (lyrInfo) { 
			lyrInfo.style.visibility = hit ? 'visible' : '';
		}
	}
}

/*===== end of geoJSON functions =====*/


//toggle layer info
function lyrInfo(gjFile,name,tempNr) {
	const date = featCols[gjFile].date ?? `<b>date not in geojson file</b>`;
	const count = featCols[gjFile].count ?? `<b>count not in geojson file</b>`;
	const info = ui.lin_templates[tempNr-1].replace('#name',name).replace('#date',date).replace('#count',count);
	toggleXbox(info);
}

//toggle side menu info
function sMenuInfo(infoNr) {
	const info = ui.sim_menuInfo[infoNr-1];
	toggleXbox(info);
}

//toggle display of Xbox
function toggleXbox(info = '') {
	let xBox = $I("xBox");
	if (xBox.style.opacity == 0) { //show
		$I("xBoxText").innerHTML = `<span class='floatR'>&nbsp;&#10060;&nbsp;</span>${info}`;
		xBox.style.visibility = "visible";
		xBox.style.opacity = 1;
	} else { //hide
		xBox.style.opacity = 0;
		xBox.style.visibility = "hidden";
	}
}

//calculate distance (haversine)
function distance(lat1,lng1,lat2,lng2) {
	const R = 6371e3; //earth radius in meters
	const rad = Math.PI/180; //1 radian
	const φ1 = lat1 * rad; //φ, λ in radians
	const φ2 = lat2 * rad;
	const Δφ = (lat2-lat1) * rad;
	const Δλ = (lng2-lng1) * rad;
	const a = Math.pow(Math.sin(Δφ/2),2) + Math.cos(φ1) * Math.cos(φ2) * Math.pow(Math.sin(Δλ/2),2);
	const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
	return R * c; // in metres
}

//get map zoom factor based on radius in km. Max. 16
function getZoom(radius) {
	return radius < 1 ? 16 : 15 - Math.round(Math.log2(radius));
}

//copy flyto URL to clipboard
function makeFlyToUrl(lat,lng) { //example: https://www.mysite.eu/pindown/?flyto=55.465591,8.702750,Z18,T4
	let curUrl = window.location.href;
	curUrl += curUrl.includes('?') ? '&' : '?';
	navigator.clipboard.writeText(`${curUrl}flyto=${Number(lat).toFixed(6)},${Number(lng).toFixed(6)},Z${cf._ftZoom},T${$I('fType').value}`);
}

//open new page with button URL (Marker Info box)
function miButton(butNr,lat,lng) {
	const butDefs = [cf._mInfoBut1, cf._mInfoBut2, cf._mInfoBut3];
	const butPars = butDefs[butNr].split(';');
	const radius = S.rRad != '0' ? S.rRad * 1000 : cf._mInfoRad; //meters
	const url = butPars[2].replace(/#lat/g,lat).replace(/#lng/g,lng).replace(/#rad/g,radius).replace(/#per/g,cf._eventDays); //substitude vars
	const target = butPars[3] ?? '_blank';
	window.open(url,target);
}

//open new page with button URL (Map Control box)
function mcEvtButton() {
	const butPars = cf._mCtrlBut.split(';');
	const radius = S.rRad * 1000; //meters
	const days = $I('evtDays').value;
	const url = butPars[2].replace(/#lat/g,S.mLat).replace(/#lng/g,S.mLon).replace(/#rad/g,radius).replace(/#per/g,days); //substitude vars
	const target = butPars[3] ?? '_blank';
	window.open(url,target);
}

//go to google for route directions
function rteDirect(latDest,lngDest) {
	const lat = $I('cLat').value.trim();
	const lng = $I('cLng').value.trim();
	if (!validLatLng(lat,lng)) { return false; }
	const url = cf._gmUrlDir + (gWatch ? '' : lat + ',' + lng) + '/' + latDest + ',' + lngDest;
	window.open(url,'_blank');
}

//modal functions
function showModal(modal,name) {
	const pattern = /(jpg|gif|png)$/i;
	const cButton = `<span class='modCls' onclick=\"hide('modal');\">&#x274C;</span>`;
	if (pattern.test(modal)) { //image
		$I('modal').innerHTML = `${cButton}<img class='modX modImg' src='${modal}' alt='PinDown image'>`;
	} else { //iframe
		$I('modal').innerHTML = `${cButton}<iframe class='modX modIfr' src='${modal}'></iframe>`;
	}
	$I('modal').style.display = "block";
}

//show info box after a marker click
function showIbox(obj,mID = 0) {
	if (typeof obj === 'object') { //Leaflet marker object
		markerI = obj.target.index;
		S.iBox = markers[markerI][0][m.img].match(/[^/]+$/) ?? markerI; //image file name or index
	} else if (typeof mID === 'number') { //index in markers array
		markerI = mID;
		S.iBox = markers[markerI][0][m.img].match(/[^/]+$/) ?? markerI; //image file name or index
	} else { //image file name
		markerI = markers.findIndex(arr => arr[0][m.img].includes(mID));
		if (markerI == -1) return; //file name not found
		S.iBox = mID; //image file name
	}
	iBoxImI = 0; //reset image index
	const mkr = markers[markerI]; //marker array
	const mkr0 = mkr[0]; //marker img
	const keys = mkr0[m.key].split('#');
	const mkrID = (keys[0] && (`mkr${keys[0]}` in mkrIDs)) ? `mkr${keys[0]}` : 'mkr0'; //first tag from list
	let buttons = '';
	const butDefs = [cf._mInfoBut1, cf._mInfoBut2, cf._mInfoBut3];
	for (let i = 0; i < 3; i++) { //process extra buttons
		const butPars = butDefs[i].split(';');
		if (butPars) {
			buttons += `<a class='button' title='${butPars[1]}' href='#' onclick='miButton(${i},${mkr0[m.lat]},${mkr0[m.lng]});'>${butPars[0]}</a> `;
		}
	}
	buttons += `<a id='ibDirect' class='button' title='${ui.inf_direct_title}' href='#' onclick='rteDirect(${mkr0[m.lat]},${mkr0[m.lng]});'>${ui.inf_direct}</a>`; //add route dir button
	const imgModal = mkr0[m.img] ? `title='${ui.lst_img_title}' onclick=\"showModal('${mkr0[m.img]}')\"` : '';
	let mkrTnl = `<img id='ibImage' class='${mkr0[m.tnl] ? 'iBoxImg' : 'iBoxMkr'}' src='${mkr0[m.tnl] ? mkr0[m.tnl] : 'markers/' + mkrIDs[mkrID]}' ${imgModal}>`;
	const lrArrows = mkr.length > 1 ? `<div id='ibArrows'class='arrows'><span>&nbsp;&emsp;&nbsp;</span><span onclick="iBoxLR('r')";>&nbsp;&#9654;&nbsp;</span></div>` : '';
	const mType = (keys[0] && _key1Val[keys[0]]) ? `<p>${_key1Val[keys[0]].split(";",1)[0]}</p>` : ''
	let info = `<div class='name'>${mkr0[m.nam]}</div>
${mType}
${(mkr0[m.sub] ? `<p>${mkr0[m.sub]}</p>` : '')}
<div class='buttons alignC'>${buttons}</div>
${lrArrows}
${mkrTnl}
<p id='ibSubTitle' class='floatC'></p>
<div class='scroll' ontouchmove='event.stopPropagation();'>
<div id='ibText'>${mkr0[m.txt].replace(/(geoJSON\('[^']{1,40}'),\d\d?\)"/g,'$1)"')}</div><br>
</div>\n`; //end of scrollarea
	$I('iBoxBody').innerHTML = info;
	$I('iBox').style.display = "block";
	$I('iBox').style.zIndex = 1000;
}

//called by ◄ and ► in Marker Info box
function iBoxLR(lr) {
	iBoxImI = lr == 'r' ? iBoxImI + 1 : iBoxImI - 1; //update image index
	const mkr = markers[markerI]; //marker array
	const mkr0 = mkr[0]; //marker img
	const mkrX = mkr[iBoxImI]; //~ img
	$I('ibSubTitle').innerHTML = iBoxImI ? mkrX[m.nam] : '';
	const mkrD = (mkrX[m.lat] && mkrX[m.lng]) ? mkrX : mkr0;
	$I('ibDirect').onclick = function() { rteDirect(mkrD[m.lat],mkrD[m.lng]); };
	const arrowL = iBoxImI > 0 ? `<span onclick="iBoxLR('l')";>&nbsp;&#9664;&nbsp;</span>` : `<span>&nbsp;&emsp;&nbsp;</span>`;
	const arrowR = iBoxImI < (mkr.length - 1) ? `<span onclick="iBoxLR('r')";>&nbsp;&#9654;&nbsp;</span>` : `<span>&nbsp;&emsp;&nbsp;</span>`;
	$I('ibArrows').innerHTML = `${arrowL}${arrowR}`;
	$I('ibImage').src = mkrX[m.tnl];
	const imgPath = mkrX[m.img];
	$I('ibImage').onclick = imgPath ? function() { showModal(imgPath); } : '';
	$I('ibText').innerHTML = (mkrX[m.txt] || mkr0[m.txt]).replace(/(geoJSON\('[^']{1,40}'),\d\d?\)"/g,'$1)"');
}

//produce HTML content for List View
function listData() {
	const len = mOnMap.length;
	let listHead = '';
	let list = '';
	let listTail = '';
	folding = 0;

	const note = (len == 0) ? ui.lst_no_markers : ((len > cf._listMax) ? ui.lst_max_items.replace('$1',cf._listMax) : '')
	if (note) { //list is empty
		$I('listInner').innerHTML = `<div class='empty'>${note}</div>\n`;
		return;
	}
	for (let i = 0; i < len; i++) { //compose list in HTML format
		const mkrIndex = mOnMap[i]; //marker number
		const imgArr = markers[mkrIndex];
//	iterate over sub-array with main (index 0) and sub-images and text
		for (let j = 0; j < imgArr.length; j++) { //compose list in HTML format
			const keys = imgArr[j][m.key].split('#');
			const mkrID = (keys[0] && ('mkr' + keys[0] in mkrIDs)) ? 'mkr' + keys[0] : 'mkr0'; //first tag from list
			const tnlPath = imgArr[j][m.tnl] || `markers/${mkrIDs[mkrID]}`;
			const imgClass = imgArr[j][m.tnl] ? 'lstImg' : 'lstMkr';
			const imgPath = imgArr[j][m.img];
			const imgModal = imgPath ? `title='${ui.lst_img_title}' onclick=\"showModal('${imgPath}')\"` : '';
			const name = imgArr[j][m.nam];
			const subject = imgArr[j][m.sub];
			const info = imgArr[j][m.txt].replace(/geoJSON\(/g,`geoJSONlv(${mkrIndex},`); //add marker number
			const k = imgArr[j][m.lat] ? j : 0; //lat/lng of sub-image or otherwise 1st image
			const ggl = cf._gmUrlLoc.replace(/#clat/gi,imgArr[k][m.lat]).replace(/#clng/gi,imgArr[k][m.lng]); //create google location link
			const imgYyyy = (cf._dateFilter && imgArr[j][m.dat]) ? ` data-year='${imgArr[j][m.dat].substr(0,4)}'` : '';
			const imgDate = imgYyyy ? `<p class='alignC'>${ID2DD(imgArr[j][m.dat])}</p>` : '';
			const multImg = (j == 0 && imgArr.length > 1) ? `<p> <button type='button'id='multImg${mkrIndex}' class='multImg' onclick=\"unfoldOne('${mkrIndex}');\">${ui.lst_more_img}</button></p>` : '';
			const mType = (keys[0] && _key1Val[keys[0]]) ? `<p>${_key1Val[keys[0]].split(";",1)[0]}</p>` : '';
			if (j == 1) { //1st of sub-images
				list += `<tbody id='group${mkrIndex}' class='fold'>`
				folding = 1;
			}
			list += `<tr class='${j == 0 ? '' : 'indent'}'${imgYyyy}>`;
			list += `<td class='floatC'><img class='${imgClass}' src='${tnlPath}' ${imgModal}>${imgDate}</td>
<td class='row2'><p class='bold'>${name}</p>${mType}${(subject ? `<p>${subject}</p>` : '')}<p><button type='button' title='${ui.lst_fly_title}' onclick=\"flyToLL('i',${imgArr[k][m.lat]},${imgArr[k][m.lng]})\">Fly to</button><button type='button' class='floatR notOnMob' title='${ui.lst_map_descr}' onclick='toggleMap(${imgArr[k][m.lat]},${imgArr[k][m.lng]})'>?</button></p><p><button type='button' title='${ui.lst_ggl_title}' onclick=\"window.open('${ggl}','_blank');\">Google maps</button></p>${multImg}</td>
<td>${info}</td>`;
			list += `</tr>\n`;
			if (j > 0 && (j + 1) == imgArr.length) { //last of sub-images
				list += `</tbody>`;
			}
		}
	}

	let head1 = folding ? `<div class='arrow arrowTop'><pre id='udArrowT' onclick="unfoldAll(0)";>▼</pre></div>` : ui.lst_photo;
	let dateFilter = cf._dateFilter ? `<input type='text' id='yyyyF' class='find' title='${ui.lst_date_filter_title}' oninput='filter()' size='7' maxlength='4' placeholder='From yyyy'>&ensp;<input type='text' id='yyyyT' class='find' title='${ui.lst_date_filter_title}' oninput='filter()' size='7' maxlength='4' placeholder='To yyyy'>` : '';
		listHead = `<a id='top' class='button'></a><div id='toTop'><a href='#top'>▲</div></a>
<table id='listTable' class='list'>
<thead>
<tr><th>${head1}</th><th>${dateFilter}</th><th>${ui.lst_descript}</th></tr>
</thead>\n`;
		listTail = `</table>\n`;
	$I('listInner').innerHTML = listHead + list + listTail; //load list
	filter();
}

//update scroll line
function scrollList() {
	const list = $I('listInner');
	$I('scrollLine').style.width = Math.trunc(362 * list.scrollTop / (list.scrollHeight - list.clientHeight)) + 'px';
}

//unfold 1 image set
function unfoldOne(tbodyID) {
	if ($I(`udArrowT`).innerHTML == '▼') {
		let tbody = $I(`group${tbodyID}`);
		if (tbody.className == 'fold') { //unfold
			tbody.className = '';
			$I(`multImg${tbodyID}`).innerHTML = ui.lst_less_img;
		} else { //fold
			tbody.className = 'fold';
			$I(`multImg${tbodyID}`).innerHTML = ui.lst_more_img;
		}
	}
}

//unfold all image sets
function unfoldAll(search) {
	let udArrowT = $I(`udArrowT`);
	let tbClass;
	let tbArrow;
	if (udArrowT.innerHTML == '▼' || search == 1) { //unfold
		udArrowT.innerHTML = '▲';
		tbClass = 'unfold';
		tbArrow = '';
	} else { //fold
		udArrowT.innerHTML = '▼';
		tbClass = 'fold';
		tbArrow = '▼';
	}
	const tbodyArr = $I('listTable').querySelectorAll("tbody.fold,tbody.unfold"); //tbody node list
	for (const tbody of tbodyArr) {
		tbody.className = tbClass;
	}	
	const multImgArr = $I('listTable').querySelectorAll(".multImg"); //multImg button node list
	for (const button of multImgArr) {
		if (tbClass == 'unfold') {
			button.style.display = 'none';
		} else {
			button.style.display = '';
			button.innerHTML = ui.lst_more_img;
		}
	}
}

//search List View texts
function filter() {
	const fText = $I('fndInp').value.trim().replace(/[-[\]{}()+.,\\\^$|]/g, '\\\\$&'); //escape all meta characters but ? and *
	if (fText && folding) {
		unfoldAll(1);
	}
	const regex = new RegExp(fText.replace(/\?/g,'.').replace('*','[^<>]+?') + '(?![^<]*>)','ig'); //+ negative look-ahead to avoid HTML tags
	const dateF = cf._dateFilter ? $I('yyyyF').value.trim() : '';
	const dateT = cf._dateFilter ? $I('yyyyT').value.trim() : '';
	const trArr = $I('listTable').querySelectorAll('tr') //tr node list
	for (const tr of trArr) {
		const tdD = tr.querySelector('td:nth-of-type(2)'); //td to search
		if (tdD === null) { continue; } //no td in header
		const tdT = tr.querySelector('td:nth-of-type(3)'); //td to search
		let hitT = 0, hitD = 0;
		if (fText) { //filter on text, highlight hits
			tdD.innerHTML = tdD.innerHTML.replace(regex, x => { hitT++; return `<abbr class='hilite'>${x}</abbr>`; });
			tdT.innerHTML = tdT.innerHTML.replace(regex, x => { hitT++; return `<abbr class='hilite'>${x}</abbr>`; });
		} else {
			hitT++; //text filter not active
		}
		if (dateF || dateT) { //filter on dates
			let yyyy = tr.dataset.year;
			if (yyyy && yyyy >= dateF.padEnd(4,"0") && yyyy <= dateT.padEnd(4,"9")) {
				hitD++;
			}
		} else {
			hitD++; //date filter not active
		}
		tr.style.display = (hitT && hitD) ? 'table-row' : 'none';
	}
}

/*===== drag functions =====*/

var theBox,boxID,posX,posY,newPosX,newPosY;

function dragMe(elmID,e) {
	boxID = elmID; //box to drag
	theBox = $I(elmID);
	if (!e) e = window.event //if ie
	posX = theBox.offsetLeft - e.clientX;
	posY = theBox.offsetTop - e.clientY;
	document.onmousemove=moveMe;
	document.onmouseup=endMove;
	return false;
}

function moveMe(e) {
	if (!e) e = window.event //if ie
	newPosX = (posX + e.clientX) < 0 ? 0 : Math.min(posX + e.clientX,window.innerWidth - theBox.offsetWidth)
	theBox.style.left = newPosX + "px";
	newPosY = (posY + e.clientY) < 0 ? 0 : Math.min(posY + e.clientY,window.innerHeight - 26)
	theBox.style.top = newPosY + "px";
}

function endMove() {
	document.onmouseup = null;
	document.onmousemove = null;
	if (newPosX) {
		S[boxID + 'X'] = newPosX; //save new position
		S[boxID + 'Y'] = newPosY;
	}
}
//end drag functions

/*===== submit functions =====*/

//submit side menu
function appMenu(elm) {
	let value = elm.value;
	let url = value.substr(1);
	let win = value[0] == '0' ? '_self' : 'appTab';
	gotoApp(url,win); //go to application
}

//goto application (substitute possible coordinates)
function gotoApp(url,win) {
	url = url.replace(/#slat/gi,S.mLat);
	url = url.replace(/#slng/gi,S.mLon);
	url = url.replace(/#clat/gi,map.getCenter().lat.toFixed(6));
	url = url.replace(/#clng/gi,map.getCenter().lng.toFixed(6));
	url = url.replace(/#zoom/gi,map.getZoom());
	window.open(url,win);
}

//move center to lat/lng
function moveTo(lat,lng,zoom = 1) {
	S.rRad = $I('rRad').value;
	S.mLat = lat;
	S.mLon = lng;
	addCircle(S.mLat,S.mLon,S.rRad);
	if (S.rRad > 0 && zoom == 1) {
		S.mZoom = getZoom(S.rRad);
	}
	addMarkers(S.mLat,S.mLon,S.rRad);
	if ((S.rRad > 0 && zoom == 1) || zoom == 3) {
		setView(lat,lng,S.mZoom);
	} else {
		setView(lat,lng);
	}
}

//submit Map Control location form
async function mlSubmit() {
	const place = $I('place').value.replace(/[&<>";{}\\]/g,''); //sanitize place
	$I('place').value = place;
	if (!place) { alert("Please enter a location name or address!"); return; }
	const cCodes = cf._countryCodes !== '' ? `&countrycodes=${cf._countryCodes.replaceAll(' ','')}` : '';
	const response = await fetch(`https://nominatim.openstreetmap.org/search?q=${place}${cCodes}&addressdetails=1&format=json`, { //OSM geocoder request
		method: 'GET',
		headers: { 'Content-Type':'application/json' },
	});
	if (!response.ok) { alert(`HTTP error! status: ${response.status}`); }
	const result = await response.json();
	if (result.length === 1) { //1 location found
		llSubmit(result[0].lat,result[0].lon);
	} else if (result.length > 1) { //> 1 location found
		let places = '';
		let count = 0;
		let curName = '';
		for (let i = 0; i < Math.min(9, result.length); i++) {
			let disName = result[i].display_name.split(/\d{4}/)[0]; //remove postcode + rest
			disName = disName.match(/^([^,$]+,\s?){2,3}[^,$]+/)[0]; //take first 4 CSV parts
			disName = disName.replace(/,\s*$/, ""); //remove last comma
			let locName = `${result[i].addresstype}, ${disName}`;
			if (locName != curName) { //don't add duplicates
				places += `<li onclick='llSubmit(${result[i].lat},${result[i].lon});'>${locName}</li>`;
			}
			curName = locName;
		}
		if (result.length > 9) {
			places += `<li>• • • • ${ui.map_more_results}</li>`;
		}
		$I('locList').innerHTML = places;
	} else { //location not found
		alert("Location not found!");
	}
}

//submit location list option
function llSubmit(lat,lng) {
	S.mLat = $I('cLat').value = Number(lat).toFixed(6);
	S.mLon = $I('cLng').value = Number(lng).toFixed(6);
	$I('locList').innerHTML = '';
	if ($I('locFly').checked) { //fly to
		flyToLL('l',S.mLat,S.mLon);
	} else { //move to
		moveTo(S.mLat,S.mLon);
	}
}

//submit Map Control coordinates form
function mcSubmit(radSet) {
	let lat = $I('cLat').value.trim();
	let lng = $I('cLng').value.trim();
	if (!validLatLng(lat,lng)) { alert (ui.map_invalid_latlng); return false; }
	S.mLat = lat;
	S.mLon = lng;
	if ($I('cooFly').checked && !radSet) { //fly to
		flyToLL('c',lat,lng);
	} else { //move to
		moveTo(lat,lng,radSet);
		$I('mcEvents').style.display = S.rRad > 0 ? 'inline' : 'none';
	}
}

//reset Map Control navigation form
function mnReset() {
	S.mLat = $I('cLat').value = cf._defLat; //map center
	S.mLon = $I('cLng').value = cf._defLng; //map center
	$I('place').value = '';
	S.rRad = $I('rRad').value = cf._defRadius;
	$I('locFly').checked = false;
	$I('cooFly').checked = false;
	S.mZoom = cf._defZoom;
	$I('flyTo').selectedIndex = 0;
	moveTo(S.mLat,S.mLon,3);
	$I('mcEvents').style.display = S.rRad > 0 ? 'inline' : 'none';
}

//submit Map Control filter form
function mfSubmit(grpChange = 0) {
	if ($I('fGroup') && grpChange) { //get value and update check boxes
		const grpObj = $I('fGroup');
		S.fGroup = grpObj.value;
		const coords = grpObj.options[grpObj.selectedIndex].dataset.coords.split(',');
		if (coords[0]) {
			if (coords[2]) {
				setView(coords[0],coords[1],coords[2]);
			} else {
				setView(coords[0],coords[1]);
			}
		}
		const gjfile = grpObj.options[grpObj.selectedIndex].dataset.gjfile;
		if (gjfile) {
			$I(`gLayer`).style.display = 'inline';
			pageView('map','list');
			geoJSON(gjfile,1); //force group layer
		} else {
			$I(`gLayer`).style.display = '';
		}
	}
	hide('infoBut');
	if ($I('fType')) {
		S.fType = $I('fType').value;
		if (S.fType > 0) {
			showInfoBut(S.fType);
		}
	}
	if ($I('key2Min')) { S.key2Min = $I('key2Min').value; }
	if ($I('key2Max')) { S.key2Max = $I('key2Max').value; }
	if ($I('dateFr')) {
		const dateFr = DD2ID($I('dateFr').value);
		const dateTo = DD2ID($I('dateTo').value);
		if (dateFr === false) { alert (ui.map_invalid_sdate);	return false; }
		if (dateTo === false) { alert (ui.map_invalid_edate);	return false; }
		if (dateFr && dateTo && dateFr > dateTo) { alert (ui.map_sdate_after_edate);	return false; }
		S.dateFr = dateFr;
		S.dateTo = dateTo;
	}
	const lat = $I('cLat').value.trim();
	const lng = $I('cLng').value.trim();
	if (!validLatLng(lat,lng)) { alert (ui.map_invalid_latlng); return false; }
	S.mLat = lat;
	S.mLon = lng;
	addCircle(S.mLat,S.mLon,S.rRad);
	addMarkers(S.mLat,S.mLon,S.rRad);
}

//reset Map Control filter form
function mfReset() {
	S.fTsOA = 0;
	S.fTags.length = 0
	$N('fTsOA')[0].checked = true; //reset OR | AND
	$N('tags[]').forEach(node => { node.checked = false; }); //reset checked tags
	$I('active').style.display = '';
	if ($I('fGroup')) { $I('fGroup').value = cf._defGroup; }
	if ($I('fType')) { $I('fType').value = _key1Start; }
	if ($I('key2Min')) { $I('key2Min').value = ''; }
	if ($I('key2Max')) { $I('key2Max').value = ''; }
	if ($I('dateFr')) { $I('dateFr').value = ''; }
	if ($I('dateTo')) { $I('dateTo').value = ''; }
	mfSubmit(1);
}

//validate coordinates
function validLatLng(lat,lng) {
	if (Math.abs(Number(lat)) > 90 ||
		Math.abs(Number(lng)) > 180 ||
		lat.search(/^\s*-?\d{1,2}\.\d{2,8}\s*$/) == -1 ||
		lng.search(/^\s*-?\d{1,3}\.\d{2,8}\s*$/) == -1) {
		return false;
	}
	return true;
}

//update 'remember settings' indicator
function mcRemember() {
	const remS = $I("rememSet").checked ? '1' : '0';
	localStorage.setItem("remS",remS);
}

/*===== fly to functions =====*/

//fly to coordinates
function flyToLL(ft,lat,lng,zoom = cf._ftZoom) { //ft: u (url), i (img), l (location), c (coords)
	pageView('map','list');
	if (fCircle) { map.removeLayer(fCircle); } //remove previous circle
	map.flyTo([lat,lng],zoom,{animate:true,duration:3.5});
	if ('ulc'.includes(ft)) {
		S.rRad = $I('rRad').value = 0;
		addCircle(0,0,0); //remove circle
		hide('mcEvents');
	}
	if ('u'.includes(ft)) {
		S.mLat = $I('cLat').value = lat;
		S.mLon = $I('cLng').value = lng;
	}
	$I('flyTo').selectedIndex = 0;
	makeFlyToUrl(lat,lng);
	setTimeout(() => { //after the flyto animation ends
		fCircle = L.circle([lat,lng],cf._cfRadius,{color:cf._cfColor,weight:cf._cfWeight,fillColor:cf._cfFill,fillOpacity:'0.1'}).addTo(map);
	},3500);
}

//fly to menu destination
function flyToDest() {
	pageView('map','list');
	const ftArr = $I('flyTo').value.split(';');
	const ll = ftArr[0].split(','); //lat/lng
	let zoom;
	if (ftArr[2] > 0) { //set zoom factor
		zoom = getZoom(ftArr[2]);
	} else {
		zoom = Number(ftArr[1]);
	}
	map.flyTo([ll[0],ll[1]],zoom,{animate:true,duration:3.5});
	S.mLat = $I('cLat').value = ll[0];
	S.mLon = $I('cLng').value = ll[1];
	S.rRad = $I('rRad').value = ftArr[2];
	S.fType = $I('fType').value = ftArr[3]; 
	$I('mcEvents').style.display = S.rRad > 0 ? 'inline' : 'none';
	addMarkers(ll[0],ll[1],S.rRad);
	if (S.rRad > 0) { //radius
		setTimeout(() => { //after the flyto animation ends
			addCircle(ll[0],ll[1],S.rRad);
		},3500);
	}
}

/*===== end of fly-to functions =====*/

/*===== locate functions =====*/

function geoLocation() {
	if (gWatch) { //clear watch
		navigator.geolocation.clearWatch(gWatchID);
		if (gCircle) { map.removeLayer(gCircle); }
		$I('gpsBut').innerHTML = ui.nav_loc_off;
		$I('gpsBar').innerHTML = '';
		hide('gpsBar');
		map.removeLayer(gMarker);
		gWatch = false;
		gCrOnce = false;
		return;
	}
	if (navigator.geolocation) { //activate watch
		const options = {enableHighAccuracy: true, timeout: 27000, maximumAge: 30000};
		navigator.geolocation.getCurrentPosition(showGeoPos,geoPosError,options);
		gWatchID = navigator.geolocation.watchPosition(showGeoPos,geoPosError,options);
		$I('gpsBut').innerHTML = ui.nav_loc_on;
		closeUdR('g'); //remove user pins
		$I('gpsBar').style.display = 'inline-block';
		gWatch = true;
	} else { 
		alert("Geolocation is not supported by this browser");
	}
}

function showGeoPos(position) {
	const lat = position.coords.latitude.toFixed(6); 
	const lng = position.coords.longitude.toFixed(6);
	const radius = Math.round(position.coords.accuracy / 2);
	const d = new Date();
	const puText = `GPS: ${ui.nav_accuracy}: ${radius} ${ui.nav_meter} - autocenter: ${(gAutoCr ? 'on' : 'off')} - ${d.toLocaleTimeString()}`;
	$I('gpsBar').innerHTML = puText;
	if (gCircle) { map.removeLayer(gCircle); }
	if (gMarker) { map.removeLayer(gMarker); }
	gCircle = L.circle([lat,lng],radius,{color:cf._crColor,weight:cf._crWeight,fillColor:cf._crFill,fillOpacity:'0.1'}).addTo(map);
	gMarker = L.marker([lat,lng],{icon:mkrIcons[cf._gMkrName], pane:'gpsPane'}).addTo(map);
	gMarker.bindTooltip(ui.nav_tooltip);
	gMarker.on('click',autoCr);
	if (gAutoCr || !gCrOnce) {
		if (!gCrOnce) {
			const zoom = getZoom(Math.round(radius / 1000)) - 1;
			setView(lat,lng,zoom);
			gCrOnce = true;
		} else {
			setView(lat,lng);
		}
	}
}

function autoCr() {
	gAutoCr = !gAutoCr;
	$I('gpsBar').innerHTML = $I('gpsBar').innerHTML.replace(/:\s(on|off)/,': ' + (gAutoCr ? 'on' : 'off'));
}

function geoPosError(error) {
	switch(error.code) {
	case error.PERMISSION_DENIED:
		alert("Locating not permitted");
		break;
	case error.POSITION_UNAVAILABLE:
		alert("Locating info unavailable");
		break;
	case error.TIMEOUT:
		alert("Locating request time-out");
		break;
	default:
		alert("Locating: unknown error");
	}
}

/*===== end of locate functions =====*/

/*===== map functions =====*/

//init marker icons
function initMkrs() {
	for (let mkrID in mkrIDs) {
		mkrIcons[mkrID] = new L.Icon({iconUrl:`markers/${mkrIDs[mkrID]}`, shadowUrl:'images/marker-shadow.png', iconSize:[25,40], iconAnchor:[12,40], popupAnchor:[1,-34], shadowSize:[40,40]});
	}
	rteNode = new L.Icon({iconUrl:`markers/node.png`, iconSize:[9,9], iconAnchor:[4,4]});
}

//set map view (center + zoom)
function setView(lat,lng,zoom) {
//	S.mLat = lat;
//	S.mLon = lng;
	if (zoom) {
		map.setView([lat,lng],zoom);
//		S.mZoom = zoom;
	} else {
		map.setView([lat,lng]);
	}
}

/*===== user route tool =====*/

//copy clicked lat/lng to lat/long fields and measure distance
function copyLatLng(e) {
	const lat = $I('cLat').value = e.latlng.lat.toFixed(6);
	const lng = $I('cLng').value = e.latlng.lng.toFixed(6);
	if (!$I('udrOn').checked) {
		usrPins.forEach(pin => { if (pin[0]) map.removeLayer(pin[0]); });
		usrPins = [];
	}
	const uMark = L.marker([lat,lng],{icon:mkrIcons['pin'], pane:'pinPane'}).addTo(map);
	uMark.on('click', function() { removePin(uMark); });
	const length = usrPins.push([uMark,lat,lng]);
	if (length == 2) {
		usrPins[0][0].off('click');
	}
	if (length > 1) {
		usrPins[length - 2][0].setIcon(rteNode);
		updatePoly();
	}
}

function removePin(pinLayer) {
	const index = usrPins.findIndex((cLayer) => { return cLayer[0] == pinLayer });
	map.removeLayer(pinLayer);
	usrPins.splice(index,1);
	let count = usrPins.length;
	if (count == 1) {
		usrPins[0][0].on('click', function() { removePin(usrPins[0][0]); });
	}
	if (count > 0) {
		usrPins[count - 1][0].setIcon(mkrIcons['pin']);
		updatePoly();
	}
}

function updatePoly() {
	$I('disBar').style.display = 'inline-block';
	if (usrPoly) { map.removeLayer(usrPoly); }
	let latlngs = [];
	usrPins.forEach(pin => { latlngs.push([pin[1],pin[2]]); });
	let totalD = 0;
	let lastD = 0;
	for (let i = 0; i < latlngs.length; i++) {
		if (i > 0) {
			lastD = Math.round(distance(latlngs[i-1][0],latlngs[i-1][1],latlngs[i][0],latlngs[i][1]));
			totalD += lastD;
		}
	}
	usrPoly = L.polyline(latlngs, {color:'blue', weight:1}).addTo(map);
	$I('disBar').innerHTML = `${ui.map_last_dist}: ${makeD(lastD)} - ${ui.map_total_dist}: ${makeD(totalD)}`;
}

function closeUdR(action) { //action: c: from checkbox, r: from total reset
	if (action == 'r') {
		$I('udrOn').checked = false;
	}
	if (!$I('udrOn').checked) {
		if (usrPoly) {
			map.removeLayer(usrPoly);
			usrPoly = [];
		}
		usrPins.forEach(pin => { map.removeLayer(pin[0]); });
		usrPins = [];
		$I('disBar').innerHTML = '';
		hide('disBar');
	}
}

function makeD(dm) {
	if (dm < 999) {
		return dm + ' m';
	} else if (dm < 99999) {
		return (dm / 1000).toFixed(2) + ' km';
	} else if (dm < 999999) {
		return (dm / 1000).toFixed(1) + ' km';
	} else {
		return (dm / 1000).toFixed(0) + ' km';
	}
}

//save user route
function saveUdR() {
	if (usrPins.length < 2) {
		alert('No user route defined!');
		return;
	}
	const rName = $I('udrName').value != '' ? $I('udrName').value : 'user route';
	const lineC = $I('lineC').value;
	const lineW = $I('lineW').value;
	let lnglats = '';
	usrPins.forEach(pin => { if (pin[0]) lnglats += `\n      [${pin[2]},${pin[1]}],`; });
	lnglats = lnglats.slice(0,-1);
	let geojson = `{\n  "type": "Feature",\n  "geometry": {\n    "type": "LineString",\n    "coordinates": [${lnglats}\n    ]\n  },\n  "properties": {\n    "name": "${rName}",\n    "description": "User defined route",\n    "stroke": "${lineC}",\n    "stroke-opacity": 1,\n    "stroke-width": ${lineW}\n  }\n}`;
	const link = document.querySelector('a.saveRoute');
	link.setAttribute('href', 'data:text/plain;charset=utf-8,' + encodeURIComponent(geojson));
	link.setAttribute('download', `${rName}.geojson`);
	link.click();
}

//load user route
function loadFile(input) {
  let reader = new FileReader();
  reader.readAsText(input.files[0]);
  reader.onload = function() {
		const fName = input.files[0].name;
		ulRoutes[fName] = JSON.parse(reader.result); //user loaded routes
		showUlR(fName);
  };
  reader.onerror = function() {
    alert(reader.error);
  };
}

//display user loaded JSON route
function showUlR(rName) {
	if (!rLayers[rName]) { //show object
		const pane = ulRoutes[rName].pane == 'area' ? 'areaPane' : 'overlayPane';
		const colProps = ulRoutes[rName].properties ?? {};
		rLayers[rName] = L.geoJSON(ulRoutes[rName], {
			style: function (feature) {
				return getGeoStyle(feature,colProps);
			},
			onEachFeature: function (feature, layer) {
				if (feature.properties.description) {
					layer.bindPopup(feature.properties.description);
				}
				if (feature.properties.distance) {
					layer.bindTooltip(feature.properties.distance, {direction:'right', permanent:true, offset: [-4,0], className: "xKm"});
				} else if (feature.properties.name) {
					layer.bindTooltip(feature.properties.name);
				}
			},
			pointToLayer: function (feature, latlng) {
        return new L.circleMarker(latlng);
			}, pane:pane
		}).addTo(map);
	}
}

//toggle all user loaded JSON routes
function toggleUlRs() {
	if ($I('tglUlR').checked) { //switch on
		for (const rName in ulRoutes) {
			if (!rLayers[rName]) {
				showUlR(rName);
			}
		}
	} else { //switch off
		for (const rName in ulRoutes) {
			if (rLayers[rName]) {
				map.removeLayer(rLayers[rName]);
				delete rLayers[rName];
			}
		}
	}
}

/*===== end of user route tool =====*/


/*========= map functions ==========*/

//init map
function initMap() {
	map.createPane('radiusPane');
	map.createPane('areaPane');
	map.createPane('pinPane');
	map.createPane('gpsPane');
	map.getPane('radiusPane').style.zIndex = 300;
	map.getPane('areaPane').style.zIndex = 350;
	map.getPane('pinPane').style.zIndex = 650;
	map.getPane('gpsPane').style.zIndex = 660;
	addCircle(S.mLat,S.mLon,S.rRad);
	addMarkers(S.mLat,S.mLon,S.rRad);
	setView(S.mLat,S.mLon,S.mZoom);
	map.on('click',copyLatLng); //copy center coordinates
}

//add circle to map
function addCircle(lat,lng,radius) {
	if (rCircle) { map.removeLayer(rCircle); } //remove previous circle
	if (radius) {
		const radInM = radius * 1000;
		rCircle = L.circle([lat,lng],radInM,{color:cf._crColor,weight:cf._crWeight,fillColor:cf._crFill,fillOpacity:'0.1',pane:'radiusPane'}).addTo(map);
	}
}

//show / hide all markers
function toggleMarkers() {
	if ($I('mkrsOn').checked) {
		addMarkers(S.mLat,S.mLon,S.rRad);
	} else {
		map.removeLayer(clusters);
	}
}

//add markers to map
function addMarkers(cLat,cLng,radius) {
	$I('mkrsOn').checked = true;//markers switch ON
	if (clusters) { map.removeLayer(clusters); }
	mOnMap = [];
	//init filters
	let rxTags;
	let rxGroup;
	let rxStr;
	let hit;
	let j;
	if (S.fTags.length > 0) { //tags filter
		const separ = (S.fTsOA == 0) ? '}|{' : '})(?=.*{'; //AND uses lookaround!
		rxStr = S.fTags.join(separ);
		rxStr = S.fTsOA == 0 ? `{${rxStr}}` : `(?=.*{${rxStr}}).*`;
		rxTags = new RegExp(rxStr, "i");
	}
	const fText = $I('fndInp').value.trim().replace(/[-[\]{}()+.,\\\^$|]/g, '\\\\$&'); //escape all meta characters but ? and *
	const rxFind = new RegExp(fText.replace(/\?/g,'.').replace('*','[^<>]+?') + '(?![^<]*>)','ig'); //+ negative look-ahead to avoid HTML tags
	//process markers
	clusters = new L.markerClusterGroup();
	clusters.options.maxClusterRadius = cf._maxClusterRadius; //max. number of pixels covered by one cluster
	if (S.fGroup) { //prepare regex for selected group(s)
		if (S.fGroup.indexOf('&') > -1) { //AND
			rxStr = '(?=.*{' +  S.fGroup.replace(/\s*\&\s*/g, '})(?=.*{') + '}).*';
		} else { //single key or OR
			rxStr = '{' +  S.fGroup.replace(/\s*\|\s*/g, '}|{') + '}';
		}
		rxGroup = new RegExp(rxStr, "i");
	}
	const auxTest = (S.key2Min || S.key2Max || S.dateFr || S.dateTo); //filter on min/max value | from/to date
	for (let i = 0, len = markers.length; i < len; i++) { //process all markers
		const marker = markers[i][0]; //[0]: first of image set
		//== start of filter ==
		const keys = marker[m.key].split('#');
		const mkrID = (keys[0] && (`mkr${keys[0]}` in mkrIDs)) ? `mkr${keys[0]}` : 'mkr0'; //1st from keywords
		if (S.fType !== '0' && (!keys[0] || !keys.includes(S.fType))) { continue; }; //type filter
		if (S.fTags.length > 0) { //filter on user-selected tags 
			if (marker[m.tgs].search(rxTags) == -1) { continue; }
		}
		if (S.fGroup) { //filter on selected-group tags
			if (marker[m.tgs].search(rxGroup) == -1) { continue; }
		}
		if (fText) {
			hit = 0;
			for (j = 0; j < markers[i].length; j++) { //search all sub-images for this marker
				if (markers[i][j][m.nam].search(rxFind) >= 0 || markers[i][j][m.sub].search(rxFind) >= 0 || markers[i][j][m.txt].search(rxFind) >= 0) { hit++; }
			}
			if (!hit) { continue; }; //search filter (no hit)
		}
		if (auxTest) {
			if (S.key2Min) { //filter on min value
				if (_key2Type.substr(1) === 'n') { //compare numbers
					if (!Boolean(keys[1]) || isNaN(keys[1]) || Number(keys[1]) < Number(S.key2Min)) { continue; }
				} else { //compare texts
					if (!Boolean(keys[1]) || keys[1] < S.key2Min) { continue; }
				}
			}
			if (S.key2Max) { //filter on max value
				if (_key2Type.substr(1) === 'n') { //compare numbers
					if (!Boolean(keys[1]) || isNaN(keys[1]) || Number(keys[1]) > Number(S.key2Max)) { continue; }
				} else { //compare texts
					if (!Boolean(keys[1]) || keys[1] > S.key2Max) { continue; }
				}
			}
			if (S.dateFr && (!marker[m.dat] || marker[m.dat] < S.dateFr)) { continue; } //from date filter
			if (S.dateTo && (!marker[m.dat] || marker[m.dat] > S.dateTo)) { continue; } //to date filter
		}
		if (radius > 0 && distance(cLat,cLng,marker[m.lat],marker[m.lng]) > (radius * 1000)) { continue; } //outside radius filter
		//== end of filter ==
		let mkr = L.marker([marker[m.lat], marker[m.lng]],{icon:mkrIcons[mkrID]});
		if (!narWin) { //make tool tip
			let ttHtml = '', ttLabel, ttField = '', bold = " class='bold'"; //init
			for (let field in _toolTip) {
				ttLabel = _toolTip[field] ? _toolTip[field] + ': ' : '';
				switch(field) {
					case '1': ttField = marker[m.nam]; break; //name
					case '2': ttField = (keys[0] && _key1Val[keys[0]]) ? _key1Val[keys[0]].split(";",1)[0] : ''; break; //marker type
					case '3': ttField = marker[m.sub]; break; //subject
					case '4': ttField = keys[1] ? keys[1] : ''; break; //keyword 2
					case '5': ttField = marker[m.dat]; break; //date
				}
				if (ttField) {
					ttHtml += `<div${bold}>${ttLabel}${ttField}</div>`;
					ttField = bold = ''; //reset
				}
			}
			const toolTip = ttHtml + (marker[m.tnl] ? `<img src='${marker[m.tnl]}' width='150'>` : '');
			if (toolTip) {
				mkr.bindTooltip(toolTip);
			}
		}
		mkr.index = i; //add index to marker
		mkr.on('click', showIbox);
		clusters.addLayer(mkr);
		mOnMap.push(i); //save markers on map (for list view)
	}
	$I("msgArea").innerHTML = `<span class=\'inputMsg\'>${mOnMap.length} ${ui.map_markers_found}</span>`;
	map.addLayer(clusters);
	listData(); //compile List View
}

/*========= miscelaneous functions =========*/

//convert display date to ISO date
function DD2ID (date) {
	if (!date) { return ''; }
	const indexY = cf._dFormat.indexOf("y") / 2;
	const indexM = cf._dFormat.indexOf("m") / 2;
	const indexD = cf._dFormat.indexOf("d") / 2;
	const split = date.split(/[^\d]/);
	if (split.length < 3 || split[indexY].search(/^\d{4}$/) !== 0 || split[indexM].search(/^\d{2}$/) !== 0 || split[indexD].search(/^\d{2}$/) !== 0) { return false; } //invalid date
	return split[indexY]+"-"+split[indexM]+"-"+split[indexD];
}

//convert ISO date to display date
function ID2DD(date) {
	return date ? cf._dFormat.replace('y',date.substr(0,4)).replace('m',date.substr(5,2)).replace('d',date.substr(8,2)) : '';
}
